/*
 * PROLL: sched_2ep.c: Interrupt management module for PCIC.
 * $Id: sched_2ep.c,v 1.10 1999/07/20 03:10:32 zaitcev Exp $
 */
#include <linux/pci.h>
#include <general.h>
#include <romlib.h>
#include <system.h>
#include "phys_jk.h"
#include "pcic.h"

#define CPUMHZ	100			/* XXX CPU clock rate */
#define TICK_TIMER_LIMIT ((CPUMHZ*1000000/4)/HZ)
#define TIMER_IRQ   10			/* Typical for a Sun */

/*
 */
struct handsc {
	void (*ifunc)(void *);
	void *iarg;
};

/*
 */
unsigned long volatile jiffies;		/* ms? ns? */

/*
 */
static volatile unsigned char *pcicp;

static int set_bolt;			/* Tick counter limit */
static struct handsc hndv[16];

/*
 */
static unsigned int swap4(unsigned int n) {
	return (n>>24 & 0xFF) | (n>>8 & 0xFF00) | ((n&0xFF00) << 8) | (n<<24);
}

/*
 * Entries from Gero's stuff.
 */
void
set_timeout(int sec)
{
	volatile int clear;

	set_bolt = sec * HZ;
	clear = *(volatile unsigned *)(pcicp + PCI_SYS_LIMIT);
	*(volatile unsigned *)(pcicp + PCI_SYS_LIMIT) = swap4(TICK_TIMER_LIMIT);
}

int	/* 0 - not expired yet; <>0 - timer expired */
chk_timeout()
{
	static int busy = 0;

	int lim = TICK_TIMER_LIMIT;
	unsigned int clear;
	unsigned int intc;
	int n;
	struct handsc *hndp;

	if (busy) {
		printk("chk_timeout: busy\n");  /* P3 */
		return 0;
	}

	/*
	 * Watch for interrupts.
	 */
	intc = swap4(*(volatile unsigned *)(pcicp + PCI_SYS_INT_PENDING));
	if ((intc & ~(1 << TIMER_IRQ)) != 0) {	/* Ignore level 10 timer */
		n = highc(intc, 16);
		hndp = &hndv[n];
		if (hndp->ifunc != 0) {
#if 0
			printk("chk_timeout: intr 0x%x\n", intc);
#endif
			busy = 1;
			(*hndp->ifunc)(hndp->iarg);
			busy = 0;
		} else {
			printk("chk_timeout: unserviced %d pending 0x%x\n",
			    n, intc);
			for (;;) {}	/* Stop console flood */
		}
	}

	/*
	 * Check if timer expired.
	 */
	clear = swap4(*(volatile unsigned *)(pcicp + PCI_SYS_COUNTER));
	if ((clear & 0x7FFFFFFF) < lim && (clear & 0x80000000) == 0)
		return 0;
	clear = *(volatile unsigned *)(pcicp + PCI_SYS_LIMIT);
	*(volatile unsigned *)(pcicp + PCI_SYS_LIMIT) = swap4(TICK_TIMER_LIMIT);
	jiffies++;
	run_timers();
	if (set_bolt > 0) {
		set_bolt--;
		return 0;
	}
	return 1;
}

/*
 * Linux environment emulator.
 */
int request_irq(unsigned int irq, void (*func)(void *), void *arg)
{
	struct handsc *hndp;

	if (irq < 0 || irq >= 16) {
		printk("request_irq: bad irq %d\n", irq);
		return -1;
	}
	hndp = &hndv[irq];

	if (hndp->ifunc != 0) {
		printk("request_irq: busy %d\n", irq);
		return -1;
	}

	hndp->ifunc = func;
	hndp->iarg = arg;

	*(volatile unsigned *)(pcicp + PCI_SYS_INT_TARGET_MASK_CLEAR) = 1 << (irq & 0xF);
	return 0;
}

void free_irq(unsigned int irq, void *arg)
{
	struct handsc *hndp;

	if (irq < 0 || irq >= 16) {
		printk("free_irq: bad irq %d\n", irq);
		return;
	}
	hndp = &hndv[irq];

	if (hndp->ifunc == 0) {
		printk("free_irq: already free %d\n", irq);
		return;
	}
	if (hndp->iarg != arg) {
		printk("free_irq: bad arg at %d\n", irq);
		return;
	}

	hndp->ifunc = 0;
	hndp->iarg = 0;

	*(volatile unsigned *)(pcicp + PCI_SYS_INT_TARGET_MASK_SET) = 1 << (irq & 0xF);
	return;
}

/*
 */
int
sched_init()
{

	pcicp = map_io(PHYS_JK_PCIC, 256);
	if (pcicp == 0) {
		printk("sched_init: cannot map CPU intr control\n");
		return -1;
	}

	*(pcicp + PCI_COUNTER_IRQ) = PCI_COUNTER_IRQ_SET(TIMER_IRQ, 0);
	*(volatile unsigned *)(pcicp + PCI_SYS_LIMIT) = swap4(TICK_TIMER_LIMIT);

	*(volatile unsigned *)(pcicp + PCI_SYS_INT_TARGET_MASK_SET) = swap4(0x7BFE);

#if 0
	printk("sched_init: stay 4\n");
	set_timeout(4);
	while (!chk_timeout()) { }
	printk("sched_init: done 4\n");
#endif

	return 0;
}
