/*
 * Primitive le diver made in 1999 from Linux sunlance.c.
 * sunlance.c: Written 1995, 1996 by Miguel de Icaza <miguel@nuclecu.unam.mx>.
 */

#include <linux/errno.h>
/* #include <asm/byteorder.h> */	/* Used by the checksum routines */
#include <asm/psr.h>		/* PSR_PIL */
#include <system.h>		/* == <asm/system.h> */
#include <general.h>		/* __P for netpriv.h */
#include <net.h>		/* ETH_ALEN */
/* #include <asm/bitops.h> */
/* #include <asm/io.h> */
#include <dma.h>		/* dmaga */
#include <netpriv.h>
#include <romlib.h>

#include "phys_jj.h"		/* hardcoded address */

/*
 * XXX Crude
 */
struct le_dma {
	struct sparc_dma_registers *regs;
	int bursizes;
	enum dvma_rev revision;
};

/* Define: 2^4 Tx buffers and 2^4 Rx buffers */
#ifndef LANCE_LOG_TX_BUFFERS
#define LANCE_LOG_TX_BUFFERS 4
#define LANCE_LOG_RX_BUFFERS 4
#endif

#define CRC_POLYNOMIAL_BE 0x04c11db7UL  /* Ethernet CRC, big endian */
#define CRC_POLYNOMIAL_LE 0xedb88320UL  /* Ethernet CRC, little endian */

#define LE_CSR0 0
#define LE_CSR1 1
#define LE_CSR2 2
#define LE_CSR3 3

#define LE_MO_PROM      0x8000  /* Enable promiscuous mode */

#define	LE_C0_ERR	0x8000	/* Error: set if BAB, SQE, MISS or ME is set */
#define	LE_C0_BABL	0x4000	/* BAB:  Babble: tx timeout. */
#define	LE_C0_CERR	0x2000	/* SQE:  Signal quality error */
#define	LE_C0_MISS	0x1000	/* MISS: Missed a packet */
#define	LE_C0_MERR	0x0800	/* ME:   Memory error */
#define	LE_C0_RINT	0x0400	/* Received interrupt */
#define	LE_C0_TINT	0x0200	/* Transmitter Interrupt */
#define	LE_C0_IDON	0x0100	/* IFIN: Init finished. */
#define	LE_C0_INTR	0x0080	/* Interrupt or error */
#define	LE_C0_INEA	0x0040	/* Interrupt enable */
#define	LE_C0_RXON	0x0020	/* Receiver on */
#define	LE_C0_TXON	0x0010	/* Transmitter on */
#define	LE_C0_TDMD	0x0008	/* Transmitter demand */
#define	LE_C0_STOP	0x0004	/* Stop the card */
#define	LE_C0_STRT	0x0002	/* Start the card */
#define	LE_C0_INIT	0x0001	/* Init the card */

#define	LE_C3_BSWP	0x4     /* SWAP */
#define	LE_C3_ACON	0x2	/* ALE Control */
#define	LE_C3_BCON	0x1	/* Byte control */

/* Receive message descriptor 1 */
#define LE_R1_OWN       0x80    /* Who owns the entry */
#define LE_R1_ERR       0x40    /* Error: if FRA, OFL, CRC or BUF is set */
#define LE_R1_FRA       0x20    /* FRA: Frame error */
#define LE_R1_OFL       0x10    /* OFL: Frame overflow */
#define LE_R1_CRC       0x08    /* CRC error */
#define LE_R1_BUF       0x04    /* BUF: Buffer error */
#define LE_R1_SOP       0x02    /* Start of packet */
#define LE_R1_EOP       0x01    /* End of packet */
#define LE_R1_POK       0x03    /* Packet is complete: SOP + EOP */

#define LE_T1_OWN       0x80    /* Lance owns the packet */
#define LE_T1_ERR       0x40    /* Error summary */
#define LE_T1_EMORE     0x10    /* Error: more than one retry needed */
#define LE_T1_EONE      0x08    /* Error: one retry needed */
#define LE_T1_EDEF      0x04    /* Error: deferred */
#define LE_T1_SOP       0x02    /* Start of packet */
#define LE_T1_EOP       0x01    /* End of packet */
#define LE_T1_POK	0x03	/* Packet is complete: SOP + EOP */

#define LE_T3_BUF       0x8000  /* Buffer error */
#define LE_T3_UFL       0x4000  /* Error underflow */
#define LE_T3_LCOL      0x1000  /* Error late collision */
#define LE_T3_CLOS      0x0800  /* Error carrier loss */
#define LE_T3_RTY       0x0400  /* Error retry */
#define LE_T3_TDR       0x03ff  /* Time Domain Reflectometry counter */

#define TX_RING_SIZE			(1 << (LANCE_LOG_TX_BUFFERS))
#define TX_RING_MOD_MASK		(TX_RING_SIZE - 1)
#define TX_RING_LEN_BITS		((LANCE_LOG_TX_BUFFERS) << 29)

#define RX_RING_SIZE			(1 << (LANCE_LOG_RX_BUFFERS))
#define RX_RING_MOD_MASK		(RX_RING_SIZE - 1)
#define RX_RING_LEN_BITS		((LANCE_LOG_RX_BUFFERS) << 29)

#define PKT_BUF_SZ		1544
#define RX_BUFF_SIZE            PKT_BUF_SZ
#define TX_BUFF_SIZE            PKT_BUF_SZ

struct lance_rx_desc {
	unsigned short rmd0;        /* low address of packet */
	unsigned char  rmd1_bits;   /* descriptor bits */
	unsigned char  rmd1_hadr;   /* high address of packet */
	short    length;    	    /* This length is 2s complement (negative)!
				     * Buffer length
				     */
	unsigned short mblength;    /* This is the actual number of bytes received */
};

struct lance_tx_desc {
	unsigned short tmd0;        /* low address of packet */
	unsigned char  tmd1_bits;   /* descriptor bits */
	unsigned char  tmd1_hadr;   /* high address of packet */
	short length;          	    /* Length is 2s complement (negative)! */
	unsigned short misc;
};

/* The LANCE initialization block, described in databook. */
/* On the Sparc, this block should be on a DMA region     */
struct lance_init_block {
	unsigned short mode;		/* Pre-set mode (reg. 15) */
	unsigned char phys_addr[6];     /* Physical ethernet address */
	unsigned filter[2];		/* Multicast filter. */

	/* Receive and transmit ring base, along with extra bits. */
	unsigned short rx_ptr;		/* receive descriptor addr */
	unsigned short rx_len;		/* receive len and high addr */
	unsigned short tx_ptr;		/* transmit descriptor addr */
	unsigned short tx_len;		/* transmit len and high addr */
    
	/* The Tx and Rx ring entries must aligned on 8-byte boundaries. */
	struct lance_rx_desc brx_ring[RX_RING_SIZE];
	struct lance_tx_desc btx_ring[TX_RING_SIZE];
    
	char   tx_buf [TX_RING_SIZE][TX_BUFF_SIZE];
	char   pad[2];			/* align rx_buf for copy_and_sum(). */
	char   rx_buf [RX_RING_SIZE][RX_BUFF_SIZE];
};

#define libdesc_offset(rt, elem) \
((__u32)(((unsigned long)(&(((struct lance_init_block *)0)->rt[elem])))))

#define libbuff_offset(rt, elem) \
((__u32)(((unsigned long)(&(((struct lance_init_block *)0)->rt[elem][0])))))

struct le_private {
	int active;			/* initialized */
	int inst;			/* iface number */

	volatile struct lance_regs *ll;
	volatile struct lance_init_block *init_block;
	__u32 init_block_dvma;
	unsigned int irq;		/* device IRQ number    */

	int rx_new, tx_new;
	int rx_old, tx_old;

	struct le_dma *ledma;         /* If set this points to ledma    */
				      /* and arch = sun4m		*/
	int tpe;		      /* cable-selection is TPE		*/
	int auto_select;	      /* cable-selection by carrier	*/
	int burst_sizes;	      /* ledma SBus burst sizes		*/

	unsigned short busmaster_regval;
	unsigned short pio_buffer;

	struct device		 edev;	/* Inherit Ethernet */
	char lename[4];
};

#define TX_BUFFS_AVAIL(lp) ((lp->tx_old<=lp->tx_new)?\
			lp->tx_old+TX_RING_MOD_MASK-lp->tx_new:\
			lp->tx_old - lp->tx_new-1)

/* On the sparc, the lance control ports are memory mapped */
struct lance_regs {
	unsigned short rdp;			/* register data port */
	unsigned short rap;			/* register address port */
};

int sparc_lance_debug = 2;

/* The Lance uses 24 bit addresses */
/* On the Sun4c the DVMA will provide the remaining bytes for us */
/* On the Sun4m we have to instruct the ledma to provide them    */
/* Even worse, on scsi/ether SBUS cards, the init block and the
 * transmit/receive buffers are addresses as offsets from absolute
 * zero on the lebuffer PIO area. -davem
 */

#define LANCE_ADDR(x) ((long)(x) & ~0xff000000)

/*
 * XXX Turn this into PROM node when needed.
 */
struct le_info {
	/*
	 * Properties
	 */
	__u32 physaddr[3];
	int tpe;	/* 0 = auto; 1 = AUI; 2 = TPE; */
	int intr;
	/*
	 * Non-properties
	 */
	struct le_dma *ledma;
};

/*
 */
static struct le_private le_arena[1];
static struct le_info le_node0;
static struct le_dma ledma0;

/* Load the CSR registers */
static void load_csrs (struct le_private *lp)
{
	volatile struct lance_regs *ll = lp->ll;
	__u32 ib_dvma = lp->init_block_dvma;
	int leptr;

	/* This is right now because when we are using a PIO buffered
	 * init block, init_block_dvma is set to zero. -DaveM
	 */
	leptr = LANCE_ADDR (ib_dvma);

	ll->rap = LE_CSR1;
	ll->rdp = (leptr & 0xFFFF);
	ll->rap = LE_CSR2;
	ll->rdp = leptr >> 16;
	ll->rap = LE_CSR3;
	ll->rdp = lp->busmaster_regval;

	/* Point back to csr0 */
	ll->rap = LE_CSR0;
}

#define ZERO 0

/* Setup the Lance Rx and Tx rings */
/* Sets dev->tbusy */
static void lance_init_ring (struct device *dev)
{
	struct le_private *lp = (struct le_private *) dev->priv;
	volatile struct lance_init_block *ib = lp->init_block;
	__u32 ib_dvma = lp->init_block_dvma;
	__u32 aib; /* for LANCE_ADDR computations */
	int leptr;
	int i;
    
	/* This is right now because when we are using a PIO buffered
	 * init block, init_block_dvma is set to zero. -DaveM
	 */
	aib = ib_dvma;

	/* Lock out other processes while setting up hardware */
	dev->tbusy = 1;
	lp->rx_new = lp->tx_new = 0;
	lp->rx_old = lp->tx_old = 0;

	ib->mode = 0;

	/* Copy the ethernet address to the lance init block
	 * Note that on the sparc you need to swap the ethernet address.
	 * Note also we want the CPU ptr of the init_block here.
	 */
	ib->phys_addr [0] = dev->dev_addr [1];
	ib->phys_addr [1] = dev->dev_addr [0];
	ib->phys_addr [2] = dev->dev_addr [3];
	ib->phys_addr [3] = dev->dev_addr [2];
	ib->phys_addr [4] = dev->dev_addr [5];
	ib->phys_addr [5] = dev->dev_addr [4];

	if (ZERO)
		printk ("TX rings:\n");
    
	/* Setup the Tx ring entries */
	for (i = 0; i <= TX_RING_SIZE; i++) {
		leptr = LANCE_ADDR(aib + libbuff_offset(tx_buf, i));
		ib->btx_ring [i].tmd0      = leptr;
		ib->btx_ring [i].tmd1_hadr = leptr >> 16;
		ib->btx_ring [i].tmd1_bits = 0;
		ib->btx_ring [i].length    = 0xf000; /* The ones required by tmd2 */
		ib->btx_ring [i].misc      = 0;
		if (i < 3)
			if (ZERO) printk ("%d: 0x%8.8x\n", i, leptr);
	}

	/* Setup the Rx ring entries */
	if (ZERO)
		printk ("RX rings:\n");
	for (i = 0; i < RX_RING_SIZE; i++) {
		leptr = LANCE_ADDR(aib + libbuff_offset(rx_buf, i));

		ib->brx_ring [i].rmd0      = leptr;
		ib->brx_ring [i].rmd1_hadr = leptr >> 16;
		ib->brx_ring [i].rmd1_bits = LE_R1_OWN;
		ib->brx_ring [i].length    = -RX_BUFF_SIZE | 0xf000;
		ib->brx_ring [i].mblength  = 0;
		if (i < 3 && ZERO)
			printk ("%d: 0x%8.8x\n", i, leptr);
	}

	/* Setup the initialization block */
    
	/* Setup rx descriptor pointer */
	leptr = LANCE_ADDR(aib + libdesc_offset(brx_ring, 0));
	ib->rx_len = (LANCE_LOG_RX_BUFFERS << 13) | (leptr >> 16);
	ib->rx_ptr = leptr;
	if (ZERO)
		printk ("RX ptr: %8.8x\n", leptr);
    
	/* Setup tx descriptor pointer */
	leptr = LANCE_ADDR(aib + libdesc_offset(btx_ring, 0));
	ib->tx_len = (LANCE_LOG_TX_BUFFERS << 13) | (leptr >> 16);
	ib->tx_ptr = leptr;
	if (ZERO)
		printk ("TX ptr: %8.8x\n", leptr);

	/* Clear the multicast filter */
	ib->filter [0] = 0;
	ib->filter [1] = 0;
}

static int init_restart_lance (struct le_private *lp)
{
	volatile struct lance_regs *ll = lp->ll;
	int i;

	if (lp->ledma) {
		struct sparc_dma_registers *dregs = lp->ledma->regs;
		unsigned long creg;

		if (!(dregs->cond_reg & DMA_HNDL_ERROR)) {
			/* E-Cache draining */
			while (dregs->cond_reg & DMA_FIFO_ISDRAIN)
				barrier();
		}

		creg = dregs->cond_reg;
		if (lp->burst_sizes & DMA_BURST32)
			creg |= DMA_E_BURST8;
		else
			creg &= ~DMA_E_BURST8;

		creg |= (DMA_DSBL_RD_DRN | DMA_DSBL_WR_INV | DMA_FIFO_INV);

		if (lp->tpe)
			creg |= DMA_EN_ENETAUI;
		else
			creg &= ~DMA_EN_ENETAUI;
		udelay(20);
		dregs->cond_reg = creg;
		udelay(200);
	}

	ll->rap = LE_CSR0;
	ll->rdp = LE_C0_INIT;

	/* Wait for the lance to complete initialization */
	for (i = 0; (i < 100) && !(ll->rdp & (LE_C0_ERR | LE_C0_IDON)); i++)
		barrier();
	if ((i == 100) || (ll->rdp & LE_C0_ERR)) {
		printk ("LANCE unopened after %d ticks, csr0=%4.4x.\n", i, ll->rdp);
		if (lp->ledma)
			printk ("dcsr=%8.8x\n",
				(unsigned int) lp->ledma->regs->cond_reg);
		return -1;
	}

	/* Clear IDON by writing a "1", enable interrupts and start lance */
	ll->rdp = LE_C0_IDON;
	ll->rdp = LE_C0_INEA | LE_C0_STRT;

	if (lp->ledma)
		lp->ledma->regs->cond_reg |= DMA_INT_ENAB;

	return 0;
}

static int lance_rx (struct device *dev)
{
	struct le_private *lp = (struct le_private *) dev->priv;
	volatile struct lance_init_block *ib = lp->init_block;
	volatile struct lance_rx_desc *rd;
	unsigned char bits;
	int len;
	struct sk_buff *skb;

#ifdef TEST_HITS
	printk ("[");
	for (i = 0; i < RX_RING_SIZE; i++) {
		if (i == lp->rx_new)
			printk ("%s",
				ib->brx_ring [i].rmd1_bits & LE_R1_OWN ? "_" : "X");
		else
			printk ("%s",
				ib->brx_ring [i].rmd1_bits & LE_R1_OWN ? "." : "1");
	}
	printk ("]");
#endif
    
	for (rd = &ib->brx_ring [lp->rx_new];
	     !((bits = rd->rmd1_bits) & LE_R1_OWN);
	     rd = &ib->brx_ring [lp->rx_new]) {

		/* We got an incomplete frame? */
		if ((bits & LE_R1_POK) != LE_R1_POK) {
			;
		} else if (bits & LE_R1_ERR) {
			/* Count only the end frame as a rx error,
			 * not the beginning
			 */
#if 0
			if (bits & LE_R1_BUF) lp->stats.rx_fifo_errors++;
			if (bits & LE_R1_CRC) lp->stats.rx_crc_errors++;
			if (bits & LE_R1_OFL) lp->stats.rx_over_errors++;
			if (bits & LE_R1_FRA) lp->stats.rx_frame_errors++;
			if (bits & LE_R1_EOP) lp->stats.rx_errors++;
#endif
		} else {
			len = (rd->mblength & 0xfff) - 4;
			skb = dev_alloc_skb (len+2);

			if (skb == 0) {
				printk ("%s: Memory squeeze, deferring packet.\n",
					dev->name);
				rd->mblength = 0;
				rd->rmd1_bits = LE_R1_OWN;
				lp->rx_new = (lp->rx_new + 1) & RX_RING_MOD_MASK;
				return 0;
			}

			skb->dev = dev;
			skb_reserve (skb, 2);		/* 16 byte align */
			skb_put (skb, len);		/* make room */
			eth_copy_and_sum(skb,
				(unsigned char *)&(ib->rx_buf [lp->rx_new][0]),
				len, 0);
			skb->protocol = eth_type_trans (skb, dev);
			netif_rx (skb);
		}

		/* Return the packet to the pool */
		rd->mblength = 0;
		rd->rmd1_bits = LE_R1_OWN;
		lp->rx_new = (lp->rx_new + 1) & RX_RING_MOD_MASK;
	}
	return 0;
}

static int lance_tx (struct device *dev)
{
	struct le_private *lp = (struct le_private *) dev->priv;
	volatile struct lance_init_block *ib = lp->init_block;
	volatile struct lance_regs *ll = lp->ll;
	volatile struct lance_tx_desc *td;
	int i, j;
	int status;

	j = lp->tx_old;
	for (i = j; i != lp->tx_new; i = j) {
		td = &ib->btx_ring [i];

		/* If we hit a packet not owned by us, stop */
		if (td->tmd1_bits & LE_T1_OWN)
			break;
		
		if (td->tmd1_bits & LE_T1_ERR) {
			status = td->misc;
#if 0
			if (status & LE_T3_RTY)  lp->stats.tx_aborted_errors++;
			if (status & LE_T3_LCOL) lp->stats.tx_window_errors++;
#endif
			if (status & LE_T3_CLOS) {
				if (lp->auto_select) {
					lp->tpe = 1 - lp->tpe;
					printk("%s: Carrier Lost, trying %s\n",
					       dev->name, lp->tpe?"TPE":"AUI");
					/* Stop the lance */
					ll->rap = LE_CSR0;
					ll->rdp = LE_C0_STOP;
					lance_init_ring (dev);
					load_csrs (lp);
					init_restart_lance (lp);
					return 0;
				}
			}

			/* Buffer errors and underflows turn off the
			 * transmitter, restart the adapter.
			 */
			if (status & (LE_T3_BUF|LE_T3_UFL)) {
				printk ("%s: Tx: ERR_BUF|ERR_UFL, restarting\n",
					dev->name);
				/* Stop the lance */
				ll->rap = LE_CSR0;
				ll->rdp = LE_C0_STOP;
				lance_init_ring (dev);
				load_csrs (lp);
				init_restart_lance (lp);
				return 0;
			}
		} else if ((td->tmd1_bits & LE_T1_POK) == LE_T1_POK) {
			/*
			 * So we don't count the packet more than once.
			 */
			td->tmd1_bits &= ~(LE_T1_POK);

#if 0
			/* One collision before packet was sent. */
			if (td->tmd1_bits & LE_T1_EONE)
				lp->stats.collisions++;

			/* More than one collision, be optimistic. */
			if (td->tmd1_bits & LE_T1_EMORE)
				lp->stats.collisions += 2;

			lp->stats.tx_packets++;
#endif
		}
	
		j = (j + 1) & TX_RING_MOD_MASK;
	}
	lp->tx_old = j;
	return 0;
}

static void lance_interrupt(void *dev_id)
{
	struct device *dev = (struct device *)dev_id;
	struct le_private *lp = (struct le_private *)dev->priv;
	volatile struct lance_regs *ll = lp->ll;
	int csr0;

	if (dev->interrupt)
		printk ("%s: again", dev->name);
    
	dev->interrupt = 1;

	ll->rap = LE_CSR0;
	csr0 = ll->rdp;

	/* Acknowledge all the interrupt sources ASAP */
	ll->rdp = csr0 & (LE_C0_INTR | LE_C0_TINT | LE_C0_RINT);
    
	if ((csr0 & LE_C0_ERR)) {
		/* Clear the error condition */
		ll->rdp = LE_C0_BABL | LE_C0_ERR | LE_C0_MISS |
			  LE_C0_CERR | LE_C0_MERR;
	}
    
	if (csr0 & LE_C0_RINT)
		lance_rx (dev);
    
	if (csr0 & LE_C0_TINT)
		lance_tx (dev);
    
	if ((TX_BUFFS_AVAIL(lp) >= 0) && dev->tbusy) {
		dev->tbusy = 0;
		mark_bh (NET_BH);
	}

#if 0
	if (csr0 & LE_C0_BABL)
		lp->stats.tx_errors++;

	if (csr0 & LE_C0_MISS)
		lp->stats.rx_errors++;
#endif

	if (csr0 & LE_C0_MERR) {
		struct sparc_dma_registers *dregs = lp->ledma->regs;
		unsigned long tst = (unsigned long)dregs->st_addr;

		printk ("%s: Memory error, status %04x, addr %06lx\n",
			dev->name, csr0, tst & 0xffffff);

		ll->rdp = LE_C0_STOP;

		if (lp->ledma)
			lp->ledma->regs->cond_reg |= DMA_FIFO_INV;

		lance_init_ring (dev);
		load_csrs (lp);
		init_restart_lance (lp);
		dev->tbusy = 0;
	}

	ll->rdp = LE_C0_INEA;
	dev->interrupt = 0;
}

static int lance_open (struct device *dev)
{
	struct le_private *lp = (struct le_private *)dev->priv;
	volatile struct lance_regs *ll = lp->ll;
	int status = 0;

	if (request_irq(lp->irq, &lance_interrupt, (void *)dev)) {
		printk ("Lance: Can't get irq %d\n", lp->irq);
		return -EAGAIN;
	}

	/* Stop the Lance */
	ll->rap = LE_CSR0;
	ll->rdp = LE_C0_STOP;

	/* On the 4m, setup the ledma to provide the upper bits for buffers */
	if (lp->ledma)
		lp->ledma->regs->dma_test = ((__u32) lp->init_block_dvma) & 0xff000000;

	lance_init_ring (dev);
	load_csrs (lp);

	dev->tbusy = 0;
	dev->interrupt = 0;
	dev->start = 1;

	status = init_restart_lance (lp);
#if 0
	/* To emulate SunOS, we add a route to the local network */
	rt_add (RTF_UP,
		dev->pa_addr & ip_get_mask (dev->pa_addr),
		ip_get_mask (dev->pa_addr),
		0, dev, dev->mtu, 0, 0);
#endif
#if 0
	if (!status && lp->auto_select) {
		/*
		 * Build a fake network packet and send it to ourselfs.
		 */
		volatile struct lance_init_block *ib = lp->init_block;
		volatile unsigned long flush;
		unsigned char packet[ETH_ZLEN];
		struct ethhdr *eth = (struct ethhdr *)packet;
		int i, entry;

		memset(packet, 0, ETH_ZLEN);
		for (i = 0; i < 6; i++) {
			eth->h_dest[i] = dev->dev_addr[i];
			eth->h_source[i] = dev->dev_addr[i];
		}

		entry = lp->tx_new & TX_RING_MOD_MASK;
		ib->btx_ring[entry].length = (-ETH_ZLEN) | 0xf000;
		ib->btx_ring[entry].misc = 0;

		bcopy(packet, (char *)&ib->tx_buf[entry][0], ETH_ZLEN);
		ib->btx_ring[entry].tmd1_bits = (LE_T1_POK|LE_T1_OWN);
		lp->tx_new = (lp->tx_new + 1) & TX_RING_MOD_MASK;

		ll->rdp = LE_C0_INEA | LE_C0_TDMD;
		flush = ll->rdp;
	}
#endif

	return status;
}

static int lance_close (struct device *dev)
{
	struct le_private *lp = (struct le_private *) dev->priv;
	volatile struct lance_regs *ll = lp->ll;

	dev->start = 0;
	dev->tbusy = 1;

	/* Stop the card */
	ll->rap = LE_CSR0;
	ll->rdp = LE_C0_STOP;

	free_irq (lp->irq, (void *) dev);
	return 0;
}

static inline int lance_reset (struct device *dev)
{
	struct le_private *lp = (struct le_private *)dev->priv;
	volatile struct lance_regs *ll = lp->ll;
	int status;
    
	/* Stop the lance */
	ll->rap = LE_CSR0;
	ll->rdp = LE_C0_STOP;

	/* On the 4m, reset the dma too */
	if (lp->ledma) {
		printk ("resetting ledma\n");
		lp->ledma->regs->cond_reg |= DMA_RST_ENET;
		udelay (200);
		lp->ledma->regs->cond_reg &= ~DMA_RST_ENET;
		lp->ledma->regs->dma_test = ((__u32) lp->init_block_dvma) & 0xff000000;
	}
	lance_init_ring (dev);
	load_csrs (lp);
	dev->trans_start = jiffies;
	dev->interrupt = 0;
	dev->start = 1;
	dev->tbusy = 0;
	status = init_restart_lance (lp);
#if 0
	printk ("Lance restart=%d\n", status);
#endif
	return status;
}

static int lance_start_xmit (struct sk_buff *skb, struct device *dev)
{
	struct le_private *lp = (struct le_private *)dev->priv;
	volatile struct lance_regs *ll = lp->ll;
	volatile struct lance_init_block *ib = lp->init_block;
	volatile unsigned long flush;
	unsigned long flags;
	int entry, skblen, len;
	int status = 0;
	static int outs;

	/* Transmitter timeout, serious problems */
	if (dev->tbusy) {
		int tickssofar = jiffies - dev->trans_start;
	    
		if (tickssofar < 100) {
			status = -1;
		} else {
			printk ("%s: transmit timed out, status %04x, reset\n",
				dev->name, ll->rdp);
			lance_reset (dev);
		}
		return status;
	}

#if 0
	/* Block a timer-based transmit from overlapping. */
	if (test_and_set_bit (0, (void *) &dev->tbusy) != 0) {
		printk ("Transmitter access conflict.\n");
		return -1;
	}
#else
	if (dev->tbusy != 0) {
		printk ("Transmitter access conflict.\n");
		return -1;
	}
	dev->tbusy = 1;
#endif

	skblen = skb->len;

	save_and_cli(flags);

	if (!TX_BUFFS_AVAIL(lp)) {
		restore_flags(flags);
		return -1;
	}

	len = (skblen <= ETH_ZLEN) ? ETH_ZLEN : skblen;
	
	entry = lp->tx_new & TX_RING_MOD_MASK;
	ib->btx_ring [entry].length = (-len) | 0xf000;
	ib->btx_ring [entry].misc = 0;
    
	bcopy(skb->data, (char *)&ib->tx_buf [entry][0], skblen);

	/* Clear the slack of the packet, do I need this? */
	/* For a firewall its a good idea - AC */
	if (len != skblen)
		bzero((char *) &ib->tx_buf [entry][skblen], len - skblen);
    
	/* Now, give the packet to the lance */
	ib->btx_ring [entry].tmd1_bits = (LE_T1_POK|LE_T1_OWN);
	lp->tx_new = (lp->tx_new+1) & TX_RING_MOD_MASK;

	outs++;
	/* Kick the lance: transmit now */
	ll->rdp = LE_C0_INEA | LE_C0_TDMD;
	dev->trans_start = jiffies;
	dev_kfree_skb (skb);
    
	if (TX_BUFFS_AVAIL(lp))
		dev->tbusy = 0;

	/* Read back CSR to invalidate the E-Cache.
	 * This is needed, because DMA_DSBL_WR_INV is set. */
	if (lp->ledma)
		flush = ll->rdp;

	restore_flags(flags);
	return status;
}

static int 
le_init(struct le_private *le, struct le_info *info, int instance)
{
	struct device *dev = &le->edev;
	volatile struct lance_regs *ll;
	int i;

	le->inst = instance;
	le->lename[0] = 'l';
	le->lename[1] = 'e';
	le->lename[2] = instance + '0';
	le->lename[3] = '\0';

	printk ("%s: LANCE ", le->lename);
	/* Copy the IDPROM ethernet address to the device structure, later we
	 * will copy the address in the device structure to the lance
	 * initialization block.
	 */
	for (i = 0; i < 6; i++)
		printk ("%x%c", dev->dev_addr[i] = idprom[i + 2],
			i == 5 ? ' ': ':');
	printk("\n");

	/* Get the IO region */
	ll = map_io(info->physaddr[1], sizeof (struct lance_regs));
	if (ll == 0) return ENOMEM;

#if 0
	/* Make certain the data structures used by the LANCE are aligned. */
	dev->priv = (void *)(((unsigned long)dev->priv + 7) & ~7);
#endif

#if 0
	lp->sbus = sdev->my_bus;
	if (lebuffer){
		prom_apply_sbus_ranges (lebuffer->my_bus,
					&lebuffer->reg_addrs [0],
					lebuffer->num_registers,
					lebuffer);

		le->init_block = (void *)
			sparc_alloc_io (lebuffer->reg_addrs [0].phys_addr, 0,
				sizeof (struct lance_init_block), "lebuffer", 
				lebuffer->reg_addrs [0].which_io, 0);
		le->init_block_dvma = 0;

		le->pio_buffer = 1;
	} else {
#endif
		le->init_block = dvma_alloc(sizeof (struct lance_init_block),
							&le->init_block_dvma);
		le->pio_buffer = 0;
#if 0
	}
#endif
	le->busmaster_regval = (LE_C3_BSWP | LE_C3_ACON | LE_C3_BCON);
	le->ll = ll;
	le->ledma = info->ledma;
	le->irq = info->intr;

	le->burst_sizes = 0;
	if (le->ledma) {
		/* Find burst-size property for ledma */
		le->burst_sizes = info->ledma->bursizes;

		/* Get the cable-selection property */
		if (info->tpe == 0) {
			printk("le%d: using auto-carrier-detection.\n",
			    le->inst);
			le->auto_select = 1;
			le->tpe = 0;
		} else if (info->tpe == 1) {
			le->auto_select = 0;
			le->tpe = 0;
		} else {
			le->auto_select = 0;
			le->tpe = 1;
			printk("le%d: using TPE.\n", le->inst);
		}

		/* Reset ledma */
		le->ledma->regs->cond_reg |= DMA_RST_ENET;
		udelay (200);
		le->ledma->regs->cond_reg &= ~DMA_RST_ENET;
	}

	/* This should never happen. */
	if ((unsigned long)(le->init_block->brx_ring) & 0x07) {
		printk("%s: ERROR: Rx and Tx rings not on even boundary.\n",
		       dev->name);
		return ENODEV;
	}

	dev->name = le->lename;
	dev->priv = le;
	dev->open = &lance_open;
	dev->stop = &lance_close;
	dev->hard_start_xmit = &lance_start_xmit;
	ether_setup (dev);

	return 0;
}

static int ledma_init(struct le_dma *ledma)
{
	void *p;

	/* Hardcode everything for MrCoffee. */
	if ((p = map_io(PHYS_JJ_LEDMA, 0x10)) == 0) {
		printk("ledma_init: cannot map registers\n");
		return -1;
	}
	ledma->regs = p;
	ledma->bursizes = 0x3F;

	printk("dma0: ");

	switch((ledma->regs->cond_reg)&DMA_DEVICE_ID) {
	case DMA_VERS0:
		ledma->revision=dvmarev0;
		printk("Revision 0 ");
		break;
	case DMA_ESCV1:
		ledma->revision=dvmaesc1;
		printk("ESC Revision 1 ");
		break;
	case DMA_VERS1:
		ledma->revision=dvmarev1;
		printk("Revision 1 ");
		break;
	case DMA_VERS2:
		ledma->revision=dvmarev2;
		printk("Revision 2 ");
		break;
	case DMA_VERHME:
		ledma->revision=dvmahme;
		printk("HME DVMA gate array ");
		break;
	case DMA_VERSPLUS:
		ledma->revision=dvmarevplus;
		printk("Revision 1 PLUS ");
		break;
	default:
		printk("unknown dma version %x",
		       (ledma->regs->cond_reg)&DMA_DEVICE_ID);
		/* ledma->allocated = 1; */
		break;
	}
	printk("\n");
	return 0;
}

/*
 * Find all the lance cards on the system and initialize them
 */
void le_probe ()
{

	/* Hardcode everything for MrCoffee. */
	le_node0.physaddr[0] = 0;
	le_node0.physaddr[1] = PHYS_JJ_LE;
	le_node0.physaddr[3] = 4;
	le_node0.tpe = 2;
	le_node0.intr = 6;		/* 0x26 - OBIO? */

	if (le_arena[0].active) {
		printk("le_probe: no free le\n");
		return;
	}

	if (ledma_init(&ledma0) != 0) {
		return;
	}
	le_node0.ledma = &ledma0;

	if (le_init(&le_arena[0], &le_node0, 0) != 0) {
		printk("le_probe: le0 init failed\n");
		return;
	}
	le_arena[0].active = 1;
	return;
}
