/*
 * PROM interface support
 * Copyright 1996 The Australian National University.
 * Copyright 1996 Fujitsu Laboratories Limited
 * Copyright 1999 Pete A. Zaitcev
 * This software may be distributed under the terms of the Gnu
 * Public License version 2 or later
 */

#include <asm/openprom.h>
#include <general.h>
#include <romlib.h>
#include <system.h>
#include <vconsole.h>
#include "phys_jj.h"

struct property {
	char *name;
	char *value;
	int length;
};

struct node {
	struct property *properties;
	/* short */ int sibling;
	/* short */ int child;
};

static int obp_nextnode(int node);
static int obp_child(int node);
static int obp_proplen(int node, char *name);
static int obp_getprop(int node, char *name, char *val);
static int obp_setprop(int node, char *name, char *val, int len);
static char *obp_nextprop(int node, char *name);

static char obp_idprom[IDPROM_SIZE];
static struct property null_properties = { NULL, NULL, -1 };
static int prop_true = -1;

static struct property propv_root[] = {
	{"name",	"SUNW,JavaStation-1", sizeof("SUNW,JavaStation-1") },
	{"idprom",	obp_idprom, IDPROM_SIZE},
	{"banner-name", "JavaStation", sizeof("JavaStation")},
	{"compatible",	"sun4m", 6},
	{NULL, NULL, -1}
};

static int prop_iommu_reg[] = {
	0x0, 0x10000000, 0x00000300,
};
static struct property propv_iommu[] = {
	{"name",	"iommu", sizeof("iommu")},
	{"reg",		(char*)&prop_iommu_reg[0], sizeof(prop_iommu_reg) },
	{NULL, NULL, -1}
};

static int prop_sbus_ranges[] = {
	0x0, 0x0, 0x0, 0x30000000, 0x10000000,
	0x1, 0x0, 0x0, 0x40000000, 0x10000000,
	0x2, 0x0, 0x0, 0x50000000, 0x10000000,
	0x3, 0x0, 0x0, 0x60000000, 0x10000000,
	0x4, 0x0, 0x0, 0x70000000, 0x10000000,
};
static struct property propv_sbus[] = {
	{"name",	"sbus", 5},
	{"ranges",	(char*)&prop_sbus_ranges[0], sizeof(prop_sbus_ranges)},
	{NULL, NULL, -1}
};

static int prop_tcx_regs[] = {
	0x2, 0x00800000, 0x00100000,
	0x2, 0x02000000, 0x00000001,
	0x2, 0x04000000, 0x00800000,
	0x2, 0x06000000, 0x00800000,
	0x2, 0x0a000000, 0x00000001,
	0x2, 0x0c000000, 0x00000001,
	0x2, 0x0e000000, 0x00000001,
	0x2, 0x00700000, 0x00001000,
	0x2, 0x00200000, 0x00000004,
	0x2, 0x00300000, 0x0000081c,
	0x2, 0x00000000, 0x00010000,
	0x2, 0x00240000, 0x00000004,
	0x2, 0x00280000, 0x00000001,
};
static int vbporch = 0x1d;
static int hbporch = 0xa0;
static int vsync = 0x6;
static int hsync = 0x88;
static int vfporch = 0x3;
static int hfporch = 0x18;
static int pixfreq = 0x03dfd240;
static int vfreq = 0x3c;
static int height = 0x300;
static int width = 0x400;
static int linebytes = 0x400;
static int depth = 8;
static int tcx_intr[] = { 5, 0 };
static int tcx_interrupts = 5;
static struct property propv_sbus_tcx[] = {
	{"name",	"SUNW,tcx", sizeof("SUNW,tcx")},
	{"vbporch",	(char*)&vbporch, sizeof(int)},
	{"hbporch",	(char*)&hbporch, sizeof(int)},
	{"vsync",	(char*)&vsync, sizeof(int)},
	{"hsync",	(char*)&hsync, sizeof(int)},
	{"vfporch",	(char*)&vfporch, sizeof(int)},
	{"hfporch",	(char*)&hfporch, sizeof(int)},
	{"pixfreq",	(char*)&pixfreq, sizeof(int)},
	{"vfreq",	(char*)&vfreq, sizeof(int)},
	{"height",	(char*)&height, sizeof(int)},
	{"width",	(char*)&width, sizeof(int)},
	{"linebytes",	(char*)&linebytes, sizeof(int)},
	{"depth",	(char*)&depth, sizeof(int)},
	{"reg",		(char*)&prop_tcx_regs[0], sizeof(prop_tcx_regs)},
	{"tcx-8-bit",	(char*)&prop_true, 0},
	{"intr",	(char*)&tcx_intr[0], sizeof(tcx_intr)},
	{"interrupts",	(char*)&tcx_interrupts, sizeof(tcx_interrupts)},
	{"device_type",	"display", sizeof("display")},
	{NULL, NULL, -1}
};

static int prop_cs4231_reg[] = {
	0x3, 0x0C000000, 0x00000040
};
static int cs4231_interrupts = 5;
static int cs4231_intr[] = { 5, 0 };

static struct property propv_sbus_cs4231[] = {
	{"name",	"SUNW,CS4231", sizeof("SUNW,CS4231") },
	{"intr",	(char*)&cs4231_intr[0], sizeof(cs4231_intr) },
	{"interrupts",  (char*)&cs4231_interrupts, sizeof(cs4231_interrupts) },	
	{"reg",		(char*)&prop_cs4231_reg[0], sizeof(prop_cs4231_reg) },
	{"device_type", "serial", sizeof("serial") },
	{"alias",	"audio", sizeof("audio") },
	{NULL, NULL, -1}
};

static int cpu_nctx = NCTX_SWIFT;
static struct property propv_cpu[] = {
	{"name",	"STP1012PGA", sizeof("STP1012PGA") },
	{"device_type",	"cpu", 4 },
	{"mmu-nctx",	(char*)&cpu_nctx, sizeof(int)},
	{NULL, NULL, -1}
};

static int prop_obio_ranges[] = {
	0x0, 0x0, 0x0, 0x71000000, 0x01000000,
};
static struct property propv_obio[] = {
	{"name",	"obio", 5 },
	{"ranges",	(char*)&prop_obio_ranges[0], sizeof(prop_obio_ranges) },
	{NULL, NULL, -1}
};

static int prop_auxio_reg[] = {
	0x0, 0x00900000, 0x00000001,
};
static struct property propv_obio_auxio[] = {
	{"name",	"auxio", sizeof("auxio") },
	{"reg",		(char*)&prop_auxio_reg[0], sizeof(prop_auxio_reg) },
	{NULL, NULL, -1}
};

static int prop_int_reg[] = {
	0x0, 0x00e00000, 0x00000010,
	0x0, 0x00e10000, 0x00000010,
};
static struct property propv_obio_int[] = {
	{"name",	"interrupt", sizeof("interrupt")},
	{"reg",		(char*)&prop_int_reg[0], sizeof(prop_int_reg) },
	{NULL, NULL, -1}
};

static int prop_cnt_reg[] = {
	0x0, 0x00d00000, 0x00000010,
	0x0, 0x00d10000, 0x00000010,
};
static struct property propv_obio_cnt[] = {
	{"name",	"counter", sizeof("counter")},
	{"reg",		(char*)&prop_cnt_reg[0], sizeof(prop_cnt_reg) },
	{NULL, NULL, -1}
};

static int prop_eeprom_reg[] = {
	0x0, 0x00200000, 0x00002000,
};
static struct property propv_obio_eep[] = {
	{"name",	"eeprom", sizeof("eeprom")},
	{"reg",		(char*)&prop_eeprom_reg[0], sizeof(prop_eeprom_reg) },
	{"model",	"mk48t08", sizeof("mk48t08")},
	{NULL, NULL, -1}
};

static int prop_su_reg[] = {
	0x0, 0x003002f8, 0x00000008,
};
static struct property propv_obio_su[] = {
	{"name",	"su", sizeof("su")},
	{"reg",		(char*)&prop_su_reg[0], sizeof(prop_su_reg) },
	{NULL, NULL, -1}
};

static int prop_ledma_reg[] = {
	0x4, 0x08400010, 0x00000020,
};
static int prop_ledma_burst = 0x3f;
static struct property propv_sbus_ledma[] = {
	{"name",	"ledma", sizeof("ledma")},
	{"reg",		(char*)&prop_ledma_reg[0], sizeof(prop_ledma_reg) },
	{"burst-sizes",	(char*)&prop_ledma_burst, sizeof(int) },
	{NULL, NULL, -1}
};

static int prop_le_reg[] = {
	0x4, 0x08c00000, 0x00000004,
};
static int prop_le_busmaster_regval = 0x7;
static int prop_le_intr[] = { 0x26, 0x0 };
static struct property propv_sbus_ledma_le[] = {
	{"name",	"le", sizeof("le")},
	{"reg",		(char*)&prop_le_reg[0], sizeof(prop_le_reg) },
	{"busmaster-regval",	(char*)&prop_le_busmaster_regval, sizeof(int)},
	{"intr",	(char*)&prop_le_intr[0], sizeof(prop_le_intr) },
	{NULL, NULL, -1}
};

static struct node nodes[] = {
	{ &null_properties,	 1,  0 }, /* 0 = big brother of root */
	{ propv_root,		 0,  2 }, /*  1 "/" */
	{ propv_iommu,		 8,  3 }, /*  2 "/iommu" */
	{ propv_sbus,		 0,  4 }, /*  3 "/iommu/sbus" */
	{ propv_sbus_tcx,	 5,  0 }, /*  4 "/iommu/sbus/SUNW,tcx" */
	{ propv_sbus_ledma,	 7,  6 }, /*  5 "/iommu/sbus/ledma" */
	{ propv_sbus_ledma_le,	 0,  0 }, /*  6 "/iommu/sbus/ledma/le" */
	{ propv_sbus_cs4231,	 0,  0 }, /*  7 "/iommu/sbus/SUNW,CS4231 */
	{ propv_cpu,		 9,  0 }, /*  8 "/STP1012PGA" */
	{ propv_obio,		 0, 10 }, /*  9 "/obio" */
	{ propv_obio_int,	11,  0 }, /* 10 "/obio/interrupt" */
	{ propv_obio_cnt,	12,  0 }, /* 11 "/obio/counter" */
	{ propv_obio_eep,	13,  0 }, /* 12 "/obio/eeprom" */
	{ propv_obio_su,	14,  0 }, /* 13 "/obio/su" */
	{ propv_obio_auxio,	 0,  0 }, /* 14 "/obio/auxio" */
};

static struct linux_mlist_v0 totphys[MAX_BANKS];
static struct linux_mlist_v0 totmap[1];
static struct linux_mlist_v0 totavail[MAX_BANKS];

static struct linux_mlist_v0 *ptphys = totphys;
static struct linux_mlist_v0 *ptmap = totmap;
static struct linux_mlist_v0 *ptavail = totavail;

static struct linux_nodeops nodeops0 = {
        obp_nextnode,	/* int (*no_nextnode)(int node); */
        obp_child,	/* int (*no_child)(int node); */
        obp_proplen,	/* int (*no_proplen)(int node, char *name); */
        obp_getprop,	/* int (*no_getprop)(int node,char *name,char *val); */
        obp_setprop,	/* int (*no_setprop)(int node, char *name,
				 char *val, int len); */
        obp_nextprop	/* char * (*no_nextprop)(int node, char *name); */
};

static struct linux_arguments_v0 obp_arg = {
        { "le()", NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ "" },
	{ 'l', 'e' },  0, 0, 0, NULL,
	NULL
};
static struct linux_arguments_v0 *obp_argp = &obp_arg;

static void *synch_hook = NULL;
static char obp_stdin = PROMDEV_KBD;
static char obp_stdout = PROMDEV_SCREEN;

static int obp_nbgetchar(void);
static int obp_nbputchar(int ch);
static void obp_reboot(char *);
static void obp_abort(void);
static void obp_halt(void);

static void doublewalk(unsigned ptab1, unsigned va)
{
unsigned int proc_tablewalk(int ctx, unsigned int va);
unsigned int mem_tablewalk(unsigned int pa, unsigned int va);

	proc_tablewalk(0, va);
	if (ptab1 != 0) mem_tablewalk(ptab1, va);
}

static struct linux_romvec romvec0 = {
	LINUX_OPPROM_MAGIC,		/* pv_magic_cookie */
	0,				/* pv_romvers - Format selector! */
	77,				/* pv_plugin_revision */
	0x10203,			/* pv_printrev */
	{				/* pv_v0mem */
		&ptphys,		/* v0_totphys */
		&ptmap,			/* v0_prommap */
		&ptavail		/* v0_available */
	},
        &nodeops0,			/* struct linux_nodeops *pv_nodeops; */
        (void*)doublewalk, /* P3 */	/* char **pv_bootstr; */
	{
		NULL,			/* v0_devopen */
		NULL,			/* v0_devclose */
		NULL,			/* v0_rdblkdev */
		NULL,			/* v0_wrblkdev */
		NULL,			/* v0_wrnetdev */
		NULL,			/* v0_rdnetdev */
		NULL,			/* v0_rdchardev */
		NULL,			/* v0_wrchardev */
		NULL			/* v0_seekdev */
	},
        &obp_stdin,			/* char *pv_stdin */
        &obp_stdout,			/* char *pv_stdout; */
        NULL,				/* int (*pv_getchar)(void); */
        NULL,				/* void (*pv_putchar)(int ch); */
        obp_nbgetchar,			/* int (*pv_nbgetchar)(void); */
        obp_nbputchar,			/* int (*pv_nbputchar)(int ch); */
        NULL,			/* void (*pv_putstr)(char *str, int len); */
        obp_reboot,			/* void (*pv_reboot)(char *bootstr); */
        NULL,		/* void (*pv_printf)(__const__ char *fmt, ...); */
        obp_abort,			/* void (*pv_abort)(void); */
        NULL,				/* __volatile__ int *pv_ticks; */
        obp_halt,			/* void (*pv_halt)(void); */
        (void *)&synch_hook,		/* void (**pv_synchook)(void); */

#if 0
        /* Evaluate a forth string, not different proto for V0 and V2->up. */
        union {
                void (*v0_eval)(int len, char *str);
                void (*v2_eval)(char *str);
        } pv_fortheval;
#endif
	{ 0 },			/* pv_fortheval */

        &obp_argp,		/* struct linux_arguments_v0 **pv_v0bootargs; */
	NULL,			/* pv_enaddr */
	{			/* pv_v2bootargs */
        	NULL,		/* char **bootpath; */
		NULL,		/* char **bootargs; */
		NULL,		/* fd_stdin; */
		NULL,		/* fd_stdout */
	},
	{			/* pv_v2devops */
        	NULL,		/* v2_inst2pkg */
        	NULL,		/* v2_dumb_mem_alloc */
        	NULL,		/* v2_dumb_mem_free */
        	NULL,		/* v2_dumb_mmap */
        	NULL,		/* v2_dumb_munmap */
        	NULL,		/* v2_dev_open */
        	NULL,		/* v2_dev_close */
        	NULL,		/* v2_dev_read */
        	NULL,		/* v2_dev_write */
        	NULL,		/* v2_dev_seek */
        	NULL,		/* v2_wheee2 */
        	NULL,		/* v2_wheee3 */
	},
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },	/* filler[15] */
        NULL,			/* pv_setctxt */
        NULL,			/* v3_cpustart */
        NULL,			/* v3_cpustop */
        NULL,			/* v3_cpuidle */
        NULL			/* v3_cpuresume */
};

void *
init_openprom(int bankc, struct bank *bankv, unsigned hiphybas)
{
	int i;

	/*
	 * Form memory descriptors.
	 */
	for (i = 0; i < bankc; i++) {
		totphys[i].theres_more = &totphys[i+1];
		totphys[i].start_adr = (char*) bankv[i].start;
		totphys[i].num_bytes = bankv[i].length;
	}
	totphys[i-1].theres_more = 0;

	/*
	 * XXX Merged in normal PROM when full banks touch.
	 */
	for (i = 0; i < bankc; i++) {
		unsigned bankbase = bankv[i].start;
		unsigned banksize = bankv[i].length;
		if (hiphybas > bankbase &&
		    hiphybas < bankbase + banksize) {
			banksize = hiphybas - bankbase;
		}
		totavail[i].theres_more = &totavail[i+1];
		totavail[i].start_adr = (char*) bankbase;
		totavail[i].num_bytes = banksize;
	}
	totavail[i-1].theres_more = 0;

	totmap[0].theres_more = 0;
	totmap[0].start_adr = (char*) PROLBASE;
	totmap[0].num_bytes = PROLSIZE;

	/*
	 * idprom
	 */
	bcopy(idprom, obp_idprom, IDPROM_SIZE);

	return &romvec0;
}

static struct property *find_property(int node,char *name)
{
	struct property *prop = &nodes[node].properties[0];
	while (prop && prop->name) {
		if (bcmp(prop->name, name, 128) == 0) return prop;
		prop++;
	}
	return NULL;
}

static int obp_nextnode(int node)
{
	return nodes[node].sibling;
}

static int obp_child(int node)
{
	return nodes[node].child;
}

static int obp_proplen(int node, char *name)
{
	struct property *prop = find_property(node,name);
	if (prop) return prop->length;
	return -1;
}

static int obp_getprop(int node, char *name, char *value)
{
	struct property *prop;
	prop = find_property(node,name);
	if (prop) {
		memcpy(value,prop->value,prop->length);
		return prop->length;
	}
	return -1;
}

static int obp_setprop(int node, char *name, char *value, int len)
{
	return -1;
}

static char *obp_nextprop(int node,char *name)
{
	struct property *prop = find_property(node,name);
	if (prop) return prop[1].name;
	return NULL;
}

#if 0
static unsigned char calc_idprom_cksum(struct idprom *idprom)
{
        unsigned char cksum, i, *ptr = (unsigned char *)idprom;

        for (i = cksum = 0; i <= 0x0E; i++)
                cksum ^= *ptr++;

        return cksum;
}
#endif

static int obp_nbgetchar(void) {
	return -1;
}

static int obp_nbputchar(int ch) {
	extern struct vconterm dp0;
	char buf = ch;

	/* We do not use printk() in order to reduce stack depth. */
	vcon_write(&dp0, &buf, 1);
	return 0;
}

static void obp_reboot(char *str) {
	printk("rebooting (%s): not implemented, freezing\n", str);
	for (;;) {}
}

static void obp_abort() {
	printk("abort, freezing\n");
	for (;;) {}
}

static void obp_halt() {
	printk("halt, freezing\n");
	for (;;) {}
}
