/*
 * PROM interface support
 * Copyright 1996 The Australian National University.
 * Copyright 1996 Fujitsu Laboratories Limited
 * Copyright 1999 Pete A. Zaitcev
 * This software may be distributed under the terms
 * of the GNU Public License version 2 or later.
 */

#include <asm/openprom.h>
#include <general.h>
#include <romlib.h>
#include <system.h>
#include <linux/pci.h>	/* makeup address */
#include <vconsole.h>
#include "phys_jk.h"
#include <silo_arg.h>

struct property {
	char *name;
	char *value;
	int length;
};

struct node {
	struct property *properties;
	/* short */ int sibling;
	/* short */ int child;
};

static int obp_nextnode(int node);
static int obp_child(int node);
static int obp_proplen(int node, char *name);
static int obp_getprop(int node, char *name, char *val);
static int obp_setprop(int node, char *name, char *val, int len);
static char *obp_nextprop(int node, char *name);

static char obp_idprom[IDPROM_SIZE];
static struct property null_properties = { NULL, NULL, -1 };
static int prop_true = -1;

/*
 * PROM has name as "SUNW,JSIIep" and banner-name "JavaStation".
 */
static struct property propv_root[] = {
	{"name",	"SUNW,JS-NC", sizeof("SUNW,JS-NC") },
	{"idprom",	obp_idprom, IDPROM_SIZE},
	{"banner-name", "JavaStation-NC", sizeof("JavaStation-NC")},
	{"compatible",	"sun4m", 6},
	{NULL, NULL, -1}
};

/*
 * Linux "knows" properties of different CPU chips so it does not
 * need the slew of properties which a regular PROM provides.
 * clock-frequency          05f5e100 
 * psr-implementation       00000000 
 * psr-version              00000004 
 * implementation           00000000 
 * version                  00000004 
 * page-size                00001000 
 * dcache-line-size         00000010 
 * dcache-nlines            00000200 
 * dcache-associativity     00000001 
 * icache-line-size         00000020 
 * icache-nlines            00000200 
 * icache-associativity     00000001 
 * ncaches                  00000002 
 * mmu-nctx                 00000100 
 * sparc-version            00000008 
 * device_type              cpu
 * name                     MicroSPARC-IIep
 */
static int cpu_nctx = NCTX_SWIFT;
static struct property propv_cpu[] = {
	{"name",	"MicroSPARC-IIep", sizeof("MicroSPARC-IIep") },
	{"device_type",	"cpu", 4 },
	{"mmu-nctx",	(char*)&cpu_nctx, sizeof(int)},
	{NULL, NULL, -1}
};

static int prop_pci_ranges[] = {
	0x00000000, 0, 0x00000000, 0, 0x00000000, 0, 0x00000000,
	0x01000000, 0, 0x30000000, 0, 0x30000000, 0, 0x00010000,
	0x02000000, 0, 0x38000000, 0, 0x38000000, 0, 0x08000000,
};
static int prop_pci_regs[] = {
	0x0, 0x300C0000, 0x00000100,	/* Own regs */
	0x0, 0x30000000, 0x00010000,	/* I/O */
	0x0, 0x30080000, 0x00000008,	/* Configuration address */
	0x0, 0x300A0000, 0x00000008,	/* Configuration data */
};
static struct property propv_pci[] = {
	{"name",	"pci", sizeof("pci") },
	{"reg",		(char*)&prop_pci_regs[0], sizeof(prop_pci_regs)},
	{"ranges",	(char*)&prop_pci_ranges[0], sizeof(prop_pci_ranges)},
	{NULL, NULL, -1}
};

/*
 * Missing from here, present in PROM:
 *  interrupts               00000002 
 *  #size-cells              00000001 
 *  #address-cells           00000002 
 */
static int prop_ebus_regs[] = {
	0x00000000, 0x0, 0x00000000, 0x0, 0x00000000,
	0x82000010, 0x0, 0xf0000000, 0x0, 0x01000000,
	0x82000014, 0x0, 0x38800000, 0x0, 0x00800000,
};
static int prop_ebus_ranges[] = {
	0x00000000, 0x00000000, 0x02000010, 0x00000000, 0x00000000, 0x01000000,
	0x00000001, 0x01000000, 0x02000014, 0x00000000, 0x00000000, 0x00800000,
};
static struct property propv_ebus[] = {
	{"name",	"ebus", sizeof("ebus") },
	{"reg",		(char*)&prop_ebus_regs[0], sizeof(prop_ebus_regs)},
	{"ranges",	(char*)&prop_ebus_ranges[0], sizeof(prop_ebus_ranges)},
	{NULL, NULL, -1}
};

static int prop_igs_regs[] = {
	0x00000800, 0x0, 0x00000000, 0x0, 0x00000000,
	0x02000810, 0x0, 0x00000000, 0x0, 0x01000000,
};
static int prop_igs_width = 0x400;
static int prop_igs_height = 0x300;
static int prop_igs_lb = 0x400;
static int prop_igs_aa[] = {
	0x00000000, 0x00000000, 0x39000000, 0x00000000, 0x00200000,
};
static struct property propv_igs[] = {
	{"name",	"igs", sizeof("igs") },
	{"device_type",	"display", sizeof("display") },
	{"reg",		(char*)&prop_igs_regs[0], sizeof(prop_igs_regs)},
	{"width",	(char*)&prop_igs_width, sizeof(int)},
	{"height",	(char*)&prop_igs_height, sizeof(int)},
	{"linebytes",	(char*)&prop_igs_lb, sizeof(int)},
	{"assigned-addresses",	(char*)&prop_igs_aa[0], sizeof(prop_igs_aa) },
	{NULL, NULL, -1}
};

/*
 *  intr                     00000001 00000000 
 *  interrupts               00000001 
 *  compatible               SUNW,hme
 */
static int prop_hme_aa[] = {
	0x00000000, 0x00000000, 0x387F8000, 0x00000000, 0x00007020,
};
static int prop_hme_regs[] = {
	0x00000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x02000110, 0x00000000, 0x00000000, 0x00000000, 0x00007020,
};
static int prop_hme_ab = 0x30;
static int prop_hme_mfs = 0x4000;
static struct property propv_hme[] = {
	{"name",	"network", sizeof("network") },
	{"compatible",	"SUNW,hme", sizeof("SUNW,hme") },
	{"reg",		(char*)&prop_hme_regs[0], sizeof(prop_hme_regs)},
	{"address-bits", (char*)&prop_hme_ab, sizeof(int)},
	{"device_type",	"network", sizeof("network") },
	{"max-frame-size",	(char*)&prop_hme_mfs, sizeof(int) },
	{"assigned-addresses",	(char*)&prop_hme_aa[0], sizeof(prop_hme_aa) },
	{NULL, NULL, -1}
};

static int prop_sua_regs[] = {
	0x00000001, 0x003002f8, 0x00000008,
};
static struct property propv_sua[] = {
	{"name",	"su", sizeof("su") },
	{"reg",		(char*)&prop_sua_regs[0], sizeof(prop_sua_regs)},
	{NULL, NULL, -1}
};

static int prop_8042_regs[] = {
	0x00000001, 0x00300060, 0x00000008,
	0x00000001, 0x00300060, 0x00000008,
};
static struct property propv_8042[] = {
	{"name",	"8042", sizeof("8042") },
	{"reg",		(char*)&prop_8042_regs[0], sizeof(prop_8042_regs)},
	{NULL, NULL, -1}
};

static int prop_kd_regs[] = {
	0x00000000,
};
static struct property propv_kd[] = {
	{"name",	"keyboard", sizeof("keyboard") },
	{"device_type",	"serial", sizeof("serial") },
	{"keyboard",	(char*)&prop_true, 0},
	{"reg",		(char*)&prop_kd_regs[0], sizeof(prop_kd_regs)},
	{NULL, NULL, -1}
};

static int prop_ms_regs[] = {
	0x00000001,
};
static struct property propv_ms[] = {
	{"name",	"kdmouse", sizeof("kdmouse") },
	{"device_type",	"serial", sizeof("serial") },
	{"mouse",	(char*)&prop_true, 0},
	{"reg",		(char*)&prop_ms_regs[0], sizeof(prop_ms_regs)},
	{NULL, NULL, -1}
};

static struct property propv_aliases[] = {
	{"name",	"aliases", sizeof("aliases") },
	{"ttya",	"/pci/ebus/su", sizeof("/pci/ebus/su")},
	{"keyboard",	"/pci/ebus/8042/keyboard", sizeof("/pci/ebus/8042/keyboard")},
	{"mouse",	"/pci/ebus/8042/kdmouse", sizeof("/pci/ebus/8042/kdmouse")},
	{"screen",	"/pci/igs", sizeof("/pci/igs")},
	{NULL, NULL, -1}
};

static int prop_rtc_regs[] = {
	0x00000001, 0x00300070, 0x00000002,
};
static int prop_rtc_interrupts = PHYS_JK_PIN_RTC;
static struct property propv_rtc[] = {
	{"name",	"rtc", sizeof("rtc") },
	{"reg",		(char*)&prop_rtc_regs[0], sizeof(prop_rtc_regs)},
	{"interrupts",	(char*)&prop_rtc_interrupts, sizeof(int)},
	{NULL, NULL, -1}
};

/*
 * XXX FIXME: Properties are hardcoded for 8MB SIMM.
 * If we ever encounter a 16B SIMM, we need to autodetect the size.
 * Flash chips have Chip ID sequence that can be used for that.
 */
static int prop_flash8_regs[] = {
	0x00000000, 0x20400000, 0x00800000,
};
static struct property propv_flash8[] = {
	{"name",	"flash-memory", sizeof("flash-memory") },
	{"reg",		(char*)&prop_flash8_regs[0], sizeof(prop_flash8_regs)},
	{"model",	"AM29LV081", sizeof("AM29LV081") },
	{NULL, NULL, -1}
};

/*
 * Based on PROM 3.12FW4 collected for us by Varol Kaplan.
 * Later made uniform with 3.11.7 (traditional, without splash screen).
 */
static struct node nodes[] = {
	{ &null_properties,	 1,  0 }, /* 0 = big brother of root */
	{ propv_root,		 0,  2 }, /*  1 "/" */
	{ propv_aliases,	 3,  0 }, /*  2 "/aliases" */
	{ propv_cpu,		 4,  0 }, /*  3 "/MicroSPARC-IIep" */
	{ propv_pci,		13,  5 }, /*  4 "/pci" */
	{ propv_igs,		 6,  0 }, /*  5 "/pci/igs" */
	{ propv_hme,		 7,  0 }, /*  6 "/pci/network" */
	{ propv_ebus,		 0,  8 }, /*  7 "/pci/ebus" */
	{ propv_sua,		 9,  0 }, /*  8 "/pci/ebus/su@1,3002f8" (A) */
	{ propv_8042,		12, 10 }, /*  9 "/pci/ebus/8042/ */
	{ propv_kd,		11,  0 }, /* 10 "/pci/ebus/8042/keyboard" */
	{ propv_ms,		 0,  0 }, /* 11 "/pci/ebus/8042/kdmouse" */
	{ propv_rtc,		 0,  0 }, /* 12 "/pci/ebus/rtc" */
	{ propv_flash8,		 0,  0 }, /* 13 "/flash-memory" */
};

static struct linux_mlist_v0 totphys[MAX_BANKS];
static struct linux_mlist_v0 totmap[1];
static struct linux_mlist_v0 totavail[MAX_BANKS];

static struct linux_mlist_v0 *ptphys = totphys;
static struct linux_mlist_v0 *ptmap = totmap;
static struct linux_mlist_v0 *ptavail = totavail;

static struct linux_nodeops nodeops0 = {
        obp_nextnode,	/* int (*no_nextnode)(int node); */
        obp_child,	/* int (*no_child)(int node); */
        obp_proplen,	/* int (*no_proplen)(int node, char *name); */
        obp_getprop,	/* int (*no_getprop)(int node,char *name,char *val); */
        obp_setprop,	/* int (*no_setprop)(int node, char *name,
				 char *val, int len); */
        obp_nextprop	/* char * (*no_nextprop)(int node, char *name); */
};

static struct linux_arguments_v0 obp_arg = {
        { "he()", NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ "" },
	{ 'h', 'e' },  0, 0, 0, NULL,
	NULL
};
static struct linux_arguments_v0 *obp_argp = &obp_arg;

static void *synch_hook = NULL;
#if 1
static char obp_stdin = PROMDEV_KBD;
static char obp_stdout = PROMDEV_SCREEN;
#else
static char obp_stdin = PROMDEV_TTYA;
static char obp_stdout = PROMDEV_TTYA;
#endif

static int obp_nbgetchar(void);
static int obp_nbputchar(int ch);
static void obp_reboot(char *);
static void obp_abort(void);
static void obp_halt(void);

static void doublewalk(unsigned ptab1, unsigned va)
{
unsigned int proc_tablewalk(int ctx, unsigned int va);
unsigned int mem_tablewalk(unsigned int pa, unsigned int va);

	proc_tablewalk(0, va);
	if (ptab1 != 0) mem_tablewalk(ptab1, va);
}

static struct linux_romvec romvec0 = {
	LINUX_OPPROM_MAGIC,		/* pv_magic_cookie */
	0,				/* pv_romvers - Format selector! */
	77,				/* pv_plugin_revision */
	0x10203,			/* pv_printrev */
	{				/* pv_v0mem */
		&ptphys,		/* v0_totphys */
		&ptmap,			/* v0_prommap */
		&ptavail		/* v0_available */
	},
        &nodeops0,			/* struct linux_nodeops *pv_nodeops; */
        (void*)doublewalk, /* P3 */	/* char **pv_bootstr; */
	{
		NULL,			/* v0_devopen */
		NULL,			/* v0_devclose */
		NULL,			/* v0_rdblkdev */
		NULL,			/* v0_wrblkdev */
		NULL,			/* v0_wrnetdev */
		NULL,			/* v0_rdnetdev */
		NULL,			/* v0_rdchardev */
		NULL,			/* v0_wrchardev */
		NULL			/* v0_seekdev */
	},
        &obp_stdin,			/* char *pv_stdin */
        &obp_stdout,			/* char *pv_stdout; */
        NULL,				/* int (*pv_getchar)(void); */
        NULL,				/* void (*pv_putchar)(int ch); */
        obp_nbgetchar,			/* int (*pv_nbgetchar)(void); */
        obp_nbputchar,			/* int (*pv_nbputchar)(int ch); */
        NULL,			/* void (*pv_putstr)(char *str, int len); */
        obp_reboot,			/* void (*pv_reboot)(char *bootstr); */
        NULL,		/* void (*pv_printf)(__const__ char *fmt, ...); */
        obp_abort,			/* void (*pv_abort)(void); */
        NULL,				/* __volatile__ int *pv_ticks; */
        obp_halt,			/* void (*pv_halt)(void); */
        (void *)&synch_hook,		/* void (**pv_synchook)(void); */

	{ 0 },			/* pv_fortheval */

        &obp_argp,		/* struct linux_arguments_v0 **pv_v0bootargs; */
	NULL,			/* pv_enaddr */
	{			/* pv_v2bootargs */
        	NULL,		/* char **bootpath; */
		NULL,		/* char **bootargs; */
		NULL,		/* fd_stdin; */
		NULL,		/* fd_stdout */
	},
	{			/* pv_v2devops */
        	NULL,		/* v2_inst2pkg */
        	NULL,		/* v2_dumb_mem_alloc */
        	NULL,		/* v2_dumb_mem_free */
        	NULL,		/* v2_dumb_mmap */
        	NULL,		/* v2_dumb_munmap */
        	NULL,		/* v2_dev_open */
        	NULL,		/* v2_dev_close */
        	NULL,		/* v2_dev_read */
        	NULL,		/* v2_dev_write */
        	NULL,		/* v2_dev_seek */
        	NULL,		/* v2_wheee2 */
        	NULL,		/* v2_wheee3 */
	},
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },	/* filler[15] */
        NULL,			/* pv_setctxt */
        NULL,			/* v3_cpustart */
        NULL,			/* v3_cpustop */
        NULL,			/* v3_cpuidle */
        NULL			/* v3_cpuresume */
};

#define CONFIG_CMD(bus, device_fn, where) (0x80000000 | (((unsigned int)bus) << 16) | (((unsigned int)device_fn) << 8) | (where & ~3))

void *
init_openprom_silo(int bankc, struct bank *bankv, unsigned hiphybas,
    struct silo_to_proll *ap)
{
	int i;

	/*
	 * Form memory descriptors.
	 */
	for (i = 0; i < bankc; i++) {
		totphys[i].theres_more = &totphys[i+1];
		totphys[i].start_adr = (char*) bankv[i].start;
		totphys[i].num_bytes = bankv[i].length;
	}
	totphys[i-1].theres_more = 0;

	/*
	 * XXX Merged in normal PROM when full banks touch.
	 */
	for (i = 0; i < bankc; i++) {
		unsigned bankbase = bankv[i].start;
		unsigned banksize = bankv[i].length;
		if (hiphybas > bankbase &&
		    hiphybas < bankbase + banksize) {
			banksize = hiphybas - bankbase;
		}
		totavail[i].theres_more = &totavail[i+1];
		totavail[i].start_adr = (char*) bankbase;
		totavail[i].num_bytes = banksize;
	}
	totavail[i-1].theres_more = 0;

	totmap[0].theres_more = 0;
	totmap[0].start_adr = (char*) PROLBASE;
	totmap[0].num_bytes = PROLSIZE;

	/*
	 * idprom
	 */
	bcopy(idprom, obp_idprom, IDPROM_SIZE);

	/*
	 * Make up "assigned-addresses"
	 */
#if 0
	{
		unsigned where;
		unsigned devfn = 0x01;
		unsigned bus = 0;
		unsigned int base0;

		where = PCI_BASE_ADDRESS_0;
		st_bp_swap(PHYS_JK_PCI_CFA, CONFIG_CMD(bus,devfn,where));
		base0 = ld_bp_swap(PHYS_JK_PCI_CFD + (where & 4));
		printk(" A0 %x", base0);
		where = PCI_BASE_ADDRESS_1;
		st_bp_swap(PHYS_JK_PCI_CFA, CONFIG_CMD(bus,devfn,where));
		base0 = ld_bp_swap(PHYS_JK_PCI_CFD + (where & 4));
		printk(" A1 %x", base0);
		where = PCI_BASE_ADDRESS_2;
		st_bp_swap(PHYS_JK_PCI_CFA, CONFIG_CMD(bus,devfn,where));
		base0 = ld_bp_swap(PHYS_JK_PCI_CFD + (where & 4));
		printk(" A2 %x\n", base0);
	}
#endif

	/*
	 * Enable DMA master for HappyMeal.
	 * Kernel 2.3.99 does _not_ do it.
	 */
	{
		unsigned where;
		unsigned devfn = 0x01;
		unsigned bus = 0;
		unsigned int cmd;

		where = PCI_COMMAND;
		st_bp_swap(PHYS_JK_PCI_CFA, CONFIG_CMD(bus,devfn,where));
		cmd = ldh_bypass(PHYS_JK_PCI_CFD + (where & 7));
		cmd |= (PCI_COMMAND_MASTER<<8);		/* byte swapped */
		st_bp_swap(PHYS_JK_PCI_CFA, CONFIG_CMD(bus,devfn,where));
		sth_bypass(PHYS_JK_PCI_CFD + (where & 7), cmd);
	}

	/*
	 */
	if (ap != 0) {
		obp_arg.argv[1] = ap->kern_args;
	}

	return &romvec0;
}

static struct property *find_property(int node,char *name)
{
	struct property *prop = &nodes[node].properties[0];
	while (prop && prop->name) {
		if (bcmp(prop->name, name, 128) == 0) return prop;
		prop++;
	}
	return NULL;
}

static int obp_nextnode(int node)
{
	return nodes[node].sibling;
}

static int obp_child(int node)
{
	return nodes[node].child;
}

static int obp_proplen(int node, char *name)
{
	struct property *prop = find_property(node,name);
	if (prop) return prop->length;
	return -1;
}

static int obp_getprop(int node, char *name, char *value)
{
	struct property *prop;
	prop = find_property(node,name);
	if (prop) {
		memcpy(value,prop->value,prop->length);
		return prop->length;
	}
	return -1;
}

static int obp_setprop(int node, char *name, char *value, int len)
{
	return -1;
}

static char *obp_nextprop(int node,char *name)
{
	struct property *prop = find_property(node,name);
	if (prop) return prop[1].name;
	return NULL;
}

#if 0
static unsigned char calc_idprom_cksum(struct idprom *idprom)
{
        unsigned char cksum, i, *ptr = (unsigned char *)idprom;

        for (i = cksum = 0; i <= 0x0E; i++)
                cksum ^= *ptr++;

        return cksum;
}
#endif

static int obp_nbgetchar(void) {
	return -1;
}

static int obp_nbputchar(int ch) {
	extern struct vconterm dp0;
	char buf = ch;

	/* We do not use printk() in order to reduce stack depth. */
	vcon_write(&dp0, &buf, 1);
	return 0;
}

static void obp_reboot(char *str) {
	printk("rebooting (%s): not implemented, freezing\n", str);
	for (;;) {}
}

static void obp_abort() {
	printk("abort, freezing\n");
	for (;;) {}
}

static void obp_halt() {
	printk("halt, freezing\n");
	for (;;) {}
}
